import express, { Response } from 'express'
import bodyParser from 'body-parser'
import { PolicyIntegrationService } from './services/policy-integration-service'
import { IoTCatalogueService, URLManagerRequest } from './services/iot-catalogue-service'
import cors from "cors"
import swaggerUI from "swagger-ui-express"
import swaggerDocument from "./swagger.json"
import { OfferingInfo, PTCatalogueService } from './services/pt-service'
const app = express()

app.use('/api-docs', swaggerUI.serve);
app.get('/api-docs', swaggerUI.setup(swaggerDocument));

const { IOT_CATALOGUE_URL, APM_URL, IOT_CATALOGUE_TOKEN, CORS_ORIGINS, PT_URL } = process.env
const corsOrigns = [
    "http://127.0.0.1:4200",
    "http://localhost:4200",
    "https://marketplace.fame-horizon.eu",
    ...(CORS_ORIGINS?.split(",") || [])
]

app.use(cors({
    origin(requestOrigin, callback) {
        callback(null, corsOrigns)
    },
    optionsSuccessStatus: 200
}))
app.use(bodyParser.json())
const port = process.env.PORT || 8080



const errorMessages: string[] = []

if (!IOT_CATALOGUE_URL) errorMessages.push("IoT Catalogue URL not defined")
if (!IOT_CATALOGUE_TOKEN) errorMessages.push("IoT Catalogue token not defined")
if (!APM_URL) errorMessages.push("APM URL not defined")
if (!PT_URL) errorMessages.push("PT URL not defined")

if (errorMessages.length > 0) throw new Error(errorMessages.join(", "))

const failedMessage = "Failed to obtain iframe url from IoT Catalogue"

function badRequest(res: Response, msg: string = "", statusCode: number = 500) {
    res.status(statusCode)
    res.end(msg)
}



async function retrieveVisibleAssetIds(jwtToken: string | undefined): Promise<string[] | null> {
    const policyIntegrationService = new PolicyIntegrationService({ jwtToken, apmURL: APM_URL! })
    const response = await policyIntegrationService.assetMarketPlaceVisibilityController.retrieveAll()
    return response?.json || null
}

async function checkAssetId(jwtToken: string | undefined, assetId: string): Promise<boolean> {
    const policyIntegrationService = new PolicyIntegrationService({ jwtToken, apmURL: APM_URL! })
    const response = await policyIntegrationService.assetMarketPlaceVisibilityController.checkOne(assetId)
    return response?.json?.hasVisibility || false
}


app.get("/", (req, res) => {
    res.redirect("/api-docs")
})


// Generate iframe URLs used for listing elements

// Assets
app.get("/generateAssetListIframeURL", async (req, res) => {


    const origin = req.query.origin as (string | undefined)
    let itemsPerPage: number | undefined
    if (typeof req.query.itemsPerPage === "string") itemsPerPage = Number(req.query.itemsPerPage)
    const jwtToken = req.headers.authorization
    const assetIds = await retrieveVisibleAssetIds(jwtToken)

    if (assetIds === null) {
        badRequest(res, failedMessage)
        return
    }
    const iotCatService = new IoTCatalogueService({ baseURL: IOT_CATALOGUE_URL!, token: IOT_CATALOGUE_TOKEN! })
    const url = await iotCatService.generateAssetListIframe({
        assetIds,
        origin,
        itemsPerPage
    })
    if (url === null) {
        badRequest(res, failedMessage)
        return
    }
    res.end(url)
})

// Training resources
app.get("/generateTrainingResourceListIframeURL", async (req, res) => {
    const origin = req.query.origin as (string | undefined)
    const projectIds = (req.query.projectIds as (string | undefined))?.split(",")
    const iotCatService = new IoTCatalogueService({ baseURL: IOT_CATALOGUE_URL!, token: IOT_CATALOGUE_TOKEN! })
    const urlManagerRequests: URLManagerRequest[] = []
    if (projectIds) {
        const selection = projectIds.map(id => ({
            categoryId: "_project",
            elementId: id
        }))
        urlManagerRequests.push(
            {
                id: "trainingResourceList",
                json: { selection }
            }
        )
    }
    const url = await iotCatService.generateTrainingResourceListIframe(origin, urlManagerRequests)
    res.end(url)
})



// Generate iframe URLs to be used to show the element details page

// Assets
app.get("/generateAssetDetailsIframeURL/:assetId", async (req, res) => {
    const origin = req.query.origin as (string | undefined)
    const jwtToken = req.headers.authorization
    const assetId = req.params.assetId
    const hasVisibility = await checkAssetId(jwtToken, assetId)
    if (!hasVisibility) {
        const statusCode = jwtToken ? 403 : 401
        badRequest(res, "Unauthorized access", statusCode)
        return
    }
    const iotCatService = new IoTCatalogueService({ baseURL: IOT_CATALOGUE_URL!, token: IOT_CATALOGUE_TOKEN! })
    const url = await iotCatService.generateAssetIframeDetails({ assetId, origin })
    if (url === null) {
        badRequest(res, failedMessage)
        return
    }
    res.end(url)
})

// Training resource
app.get("/generateTrainingResourceDetailsIframeURL/:trainingResourceId", async (req, res) => {
    const origin = req.query.origin as (string | undefined)

    const trainingResourceId = req.params.trainingResourceId

    const iotCatService = new IoTCatalogueService({ baseURL: IOT_CATALOGUE_URL!, token: IOT_CATALOGUE_TOKEN! })
    const url = await iotCatService.generateTrainingResourceIframeDetails({ trainingResourceId, origin })
    if (url === null) {
        badRequest(res, failedMessage)
        return
    }
    res.end(url)
})

// Validation
app.get("/generateValidationDetailsIframeURL/:validationId", async (req, res) => {
    const origin = req.query.origin as (string | undefined)
    const validationId = req.params.validationId
    const iotCatService = new IoTCatalogueService({ baseURL: IOT_CATALOGUE_URL!, token: IOT_CATALOGUE_TOKEN! })
    const url = await iotCatService.generateValidationIframeDetails({ validationId, origin })
    if (url === null) {
        badRequest(res, failedMessage)
        return
    }
    res.end(url)
})


// Data endpoints

// Counts the data elements filtered per collectionName (mandatory) and by tagNames (optional)
app.get("/count", async (req, res) => {
    const { collectionName, tagNames } = req.query
    if (!collectionName) {
        badRequest(res, 'Missing query parameter "collectionName"', 400)
        return
    }
    let _tagNames: string[] | undefined
    if (typeof tagNames === "string") _tagNames = tagNames.split(",")
    const iotCatService = new IoTCatalogueService({ baseURL: IOT_CATALOGUE_URL!, token: IOT_CATALOGUE_TOKEN! })
    const count = await iotCatService.getCount(collectionName as string, _tagNames)
    res.end(count.toString())
})

// Returns asset details
app.get("/assetDetails/:assetId", async (req, res) => {
    const { assetId } = req.params
    const extraFields = req.query.extraFields as string | undefined
    const jwtToken = req.headers.authorization
    const hasVisibility = await checkAssetId(jwtToken, assetId)
    if (!hasVisibility) {
        const statusCode = jwtToken ? 403 : 401
        badRequest(res, "Unauthorized access", statusCode)
        return
    }
    const iotCatService = new IoTCatalogueService({ baseURL: IOT_CATALOGUE_URL!, token: IOT_CATALOGUE_TOKEN! })
    res.setHeader("Content-Type", "application/json")
    const data = await iotCatService.getDataElement("components", assetId, extraFields?.split(","))
    if (!data) {
        badRequest(res, "Asset not found", 404)
        return
    }
    const ptService = new PTCatalogueService(PT_URL as string, jwtToken)
    const offerings = await ptService.getAssetOfferings([assetId])
    res.json({
        ...data,
        offerings
    })
})

function getOfferingHashMap(offerings: OfferingInfo[]): { [aid: string]: OfferingInfo[] } {
    const hm: { [aid: string]: OfferingInfo[] } = {}
    for (const offering of offerings) {
        if (!hm[offering.aid]) hm[offering.aid] = []
        hm[offering.aid].push(offering)
    }
    return hm
}

// Search for assets based on a term
app.post("/searchAssets", async (req, res) => {
    if (!req.body) {
        badRequest(res, "Malformed request", 400)
        return
    }
    const jwtToken = req.headers.authorization
    const visibleAssets = await retrieveVisibleAssetIds(jwtToken)
    const iotCatService = new IoTCatalogueService({ baseURL: IOT_CATALOGUE_URL!, token: IOT_CATALOGUE_TOKEN! })
    if (visibleAssets === null) {
        badRequest(res, failedMessage)
        return
    }

    const result = await iotCatService.searchAssets(visibleAssets, req.body)
    if (result === null) {
        badRequest(res, "Failed to obtain results")
        return
    }
    if (req.body.expand) {
        const ids = result.results?.map(e => e.id)
        const ptService = new PTCatalogueService(PT_URL as string, jwtToken)
        const offerings = await ptService.getAssetOfferings(ids)
        const hm = getOfferingHashMap(offerings)
        for (const { dataElement } of (result.results || [])) {
            if (dataElement) dataElement.offerings = hm[dataElement._id]
        }
    }
    res.json(result)
})

app.listen(port)
console.log(`Added the following cors origins: ${corsOrigns.join(", ")}`)
console.log(`Server listening on port: ${port}`)


